# bookkeeping/forms.py

from django import forms
from .models import Income, Expense, Category


# ============================================================
# CATEGORY FORM
# ============================================================


class CategoryForm(forms.ModelForm):
    class Meta:
        model = Category
        fields = ["name", "category_type", "description", "is_active"]
        widgets = {
            "name": forms.TextInput(attrs={"class": "w-full border px-3 py-2 rounded"}),
            "category_type": forms.Select(
                attrs={"class": "w-full border px-3 py-2 rounded"}
            ),
            "description": forms.Textarea(
                attrs={"class": "w-full border px-3 py-2 rounded", "rows": 3}
            ),
        }


# ============================================================
# INCOME FORM
# ============================================================


class IncomeForm(forms.ModelForm):
    class Meta:
        model = Income
        fields = [
            "date",
            "description",
            "amount",
            "category",
            "client_name",
            "invoice_number",
            "notes",
        ]
        widgets = {
            "date": forms.DateInput(
                attrs={
                    "type": "date",
                    "class": "w-full border px-3 py-2 rounded",
                },
                format="%Y-%m-%d",
            ),
            "description": forms.TextInput(
                attrs={
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Brief description of the income",
                }
            ),
            "amount": forms.NumberInput(
                attrs={"step": "0.01", "class": "w-full border px-3 py-2 rounded"}
            ),
            "category": forms.Select(
                attrs={"class": "w-full border px-3 py-2 rounded"}
            ),
            "client_name": forms.TextInput(
                attrs={
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Person or business who paid you",
                }
            ),
            "invoice_number": forms.TextInput(
                attrs={
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Optional invoice reference",
                }
            ),
            "notes": forms.Textarea(
                attrs={
                    "rows": 3,
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Additional notes (optional)",
                }
            ),
        }

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop("user", None)
        super().__init__(*args, **kwargs)

        # Filter categories for income
        self.fields["category"].queryset = Category.objects.filter(
            category_type="income"
        )

        # Existing date automatically handled by Django — no overrides needed

    def save(self, commit=True):
        obj = super().save(commit=False)
        if self.user and not obj.user_id:
            obj.user = self.user
        if commit:
            obj.save()
        return obj


# ============================================================
# EXPENSE FORM
# ============================================================


class ExpenseForm(forms.ModelForm):
    class Meta:
        model = Expense
        fields = [
            "date",
            "description",
            "amount",
            "category",
            "supplier_name",
            "vat_rate",
            "vat_amount",
            "notes",
        ]
        widgets = {
            "date": forms.DateInput(
                attrs={
                    "type": "date",
                    "class": "w-full border px-3 py-2 rounded",
                },
                format="%Y-%m-%d",
            ),
            "description": forms.TextInput(
                attrs={
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Brief description of the expense",
                }
            ),
            "amount": forms.NumberInput(
                attrs={"step": "0.01", "class": "w-full border px-3 py-2 rounded"}
            ),
            "category": forms.Select(
                attrs={"class": "w-full border px-3 py-2 rounded"}
            ),
            "supplier_name": forms.TextInput(
                attrs={
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Name of supplier (optional)",
                }
            ),
            "vat_rate": forms.NumberInput(
                attrs={"step": "0.01", "class": "w-full border px-3 py-2 rounded"}
            ),
            "vat_amount": forms.NumberInput(
                attrs={"step": "0.01", "class": "w-full border px-3 py-2 rounded"}
            ),
            "notes": forms.Textarea(
                attrs={
                    "rows": 3,
                    "class": "w-full border px-3 py-2 rounded",
                    "placeholder": "Additional notes (optional)",
                }
            ),
        }

    def __init__(self, *args, **kwargs):
        self.user = kwargs.pop("user", None)
        super().__init__(*args, **kwargs)

        # Filter categories for expenses
        self.fields["category"].queryset = Category.objects.filter(
            category_type="expense"
        )
        # Existing date automatically handled by Django — no overrides needed

    def save(self, commit=True):
        obj = super().save(commit=False)
        if self.user and not obj.user_id:
            obj.user = self.user
        if commit:
            obj.save()
        return obj
