from django.db import models
from django.contrib.auth import get_user_model
from decimal import Decimal
from django.core.exceptions import ValidationError
from datetime import datetime

User = get_user_model()


class Category(models.Model):
    CATEGORY_TYPE_CHOICES = [
        ("income", "Income"),
        ("expense", "Expense"),
    ]

    name = models.CharField(max_length=100)
    category_type = models.CharField(max_length=10, choices=CATEGORY_TYPE_CHOICES)
    hmrc_category_code = models.CharField(max_length=20, blank=True)
    description = models.TextField(blank=True)
    is_active = models.BooleanField(default=True)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ["name"]
        verbose_name_plural = "Categories"

    def __str__(self):
        return f"{self.name} ({self.category_type})"


class Income(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="incomes")
    date = models.DateField(db_index=True)
    description = models.CharField(max_length=200)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    category = models.ForeignKey(
        Category, on_delete=models.PROTECT, limit_choices_to={"category_type": "income"}
    )

    client_name = models.CharField(max_length=200, blank=True)
    invoice_number = models.CharField(max_length=100, blank=True)

    quarter = models.CharField(max_length=10, db_index=True, blank=True)
    notes = models.TextField(blank=True)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ["-date"]

    def save(self, *args, **kwargs):
        self.quarter = self._calculate_quarter()
        super().save(*args, **kwargs)

    def _calculate_quarter(self):
        date = self.date
        if isinstance(date, str):
            date = datetime.strptime(date, "%Y-%m-%d").date()

        # Tax year handling
        if date.month > 4 or (date.month == 4 and date.day >= 6):
            year = date.year
        else:
            year = date.year - 1

        # Quarter breakpoints
        if (
            (date.month == 4 and date.day >= 6)
            or date.month in [5, 6]
            or (date.month == 7 and date.day <= 5)
        ):
            return f"{year}-Q1"
        elif (
            (date.month == 7 and date.day >= 6)
            or date.month in [8, 9]
            or (date.month == 10 and date.day <= 5)
        ):
            return f"{year}-Q2"
        elif (
            (date.month == 10 and date.day >= 6)
            or date.month in [11, 12]
            or (date.month == 1 and date.day <= 5)
        ):
            return f"{year}-Q3"
        else:
            return f"{year}-Q4"


class Expense(models.Model):
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="expenses")
    date = models.DateField(db_index=True)
    description = models.CharField(max_length=200)
    amount = models.DecimalField(max_digits=10, decimal_places=2)
    category = models.ForeignKey(
        Category,
        on_delete=models.PROTECT,
        limit_choices_to={"category_type": "expense"},
    )

    vat_amount = models.DecimalField(
        max_digits=10, decimal_places=2, default=Decimal("0.00")
    )
    vat_rate = models.DecimalField(
        max_digits=4, decimal_places=2, default=Decimal("0.00")
    )

    supplier_name = models.CharField(max_length=200, blank=True)

    quarter = models.CharField(max_length=10, db_index=True, blank=True)
    notes = models.TextField(blank=True)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ["-date"]

    def save(self, *args, **kwargs):
        self.quarter = self._calculate_quarter()
        super().save(*args, **kwargs)

    def clean(self):
        if self.vat_rate and self.vat_rate > 0:
            expected_vat = round(float(self.amount) * float(self.vat_rate) / 100, 2)
            actual_vat = round(float(self.vat_amount or 0), 2)
            if abs(expected_vat - actual_vat) > 0.05:
                raise ValidationError(
                    {
                        "vat_amount": (
                            f"VAT amount £{actual_vat:.2f} does not match "
                            f"{self.vat_rate:.0f}% of £{self.amount:.2f} "
                            f"(should be £{expected_vat:.2f})."
                        )
                    }
                )

    def _calculate_quarter(self):
        return Income._calculate_quarter(self)


class ProfitAndLoss(models.Model):
    class Meta:
        managed = False
        verbose_name_plural = "Profit & Loss Summary"

    def __str__(self):
        return "Profit & Loss Summary"
