# bookkeeping/views.py
from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.core.paginator import Paginator
from django.db.models import Q, Sum
import csv
from django.http import HttpResponse
from .models import Income, Expense, Category
from .forms import IncomeForm, ExpenseForm


# ===========================
# INCOME VIEWS
# ===========================


@login_required
def income_list(request):
    """List all income entries for the logged-in user."""

    qs = Income.objects.filter(user=request.user).select_related("category")

    # Search
    search = request.GET.get("search")
    if search:
        qs = qs.filter(
            Q(description__icontains=search)
            | Q(client_name__icontains=search)
            | Q(invoice_number__icontains=search)
        )

    # Quarter filter
    quarter = request.GET.get("quarter")
    if quarter:
        qs = qs.filter(quarter=quarter)

    # Category filter
    category_id = request.GET.get("category")
    if category_id:
        qs = qs.filter(category_id=category_id)

    # Date range
    date_from = request.GET.get("date_from")
    if date_from:
        qs = qs.filter(date__gte=date_from)

    date_to = request.GET.get("date_to")
    if date_to:
        qs = qs.filter(date__lte=date_to)

    # Order
    order_by = request.GET.get("order_by", "-date")
    qs = qs.order_by(order_by)

    # Totals
    total_income = qs.aggregate(total=Sum("amount"))["total"] or 0

    # Pagination
    paginator = Paginator(qs, 20)
    page = paginator.get_page(request.GET.get("page"))

    context = {
        "income_list": page,
        "total_income": total_income,
        "income_categories": Category.objects.filter(category_type="income"),
        "quarters": (
            Income.objects.filter(user=request.user)
            .values_list("quarter", flat=True)
            .distinct()
            .order_by("-quarter")
        ),
        # Filters (returned to template)
        "search_query": search,
        "filter_date_from": date_from,
        "filter_date_to": date_to,
        "filter_category": category_id,
        "filter_quarter": quarter,
        "order_by": order_by,
    }

    return render(request, "bookkeeping/income/income_list.html", context)


@login_required
def income_create(request):
    if request.method == "POST":
        form = IncomeForm(request.POST, user=request.user)
        if form.is_valid():
            obj = form.save(commit=False)
            obj.user = request.user
            obj.save()

            if "save_and_add" in request.POST:
                return redirect("bookkeeping:income_create")
            return redirect("bookkeeping:income_list")
    else:
        form = IncomeForm(user=request.user)

    return render(request, "bookkeeping/income/income_form.html", {"form": form})


@login_required
def income_detail(request, pk):
    income = get_object_or_404(Income, pk=pk, user=request.user)
    return render(request, "bookkeeping/income/income_detail.html", {"income": income})


@login_required
def income_edit(request, pk):
    income = get_object_or_404(Income, pk=pk, user=request.user)

    if request.method == "POST":
        form = IncomeForm(request.POST, instance=income, user=request.user)
        if form.is_valid():
            obj = form.save(commit=False)
            obj.user = request.user
            obj.save()
            return redirect("bookkeeping:income_detail", pk=income.pk)
    else:
        form = IncomeForm(instance=income, user=request.user)

    return render(
        request,
        "bookkeeping/income/income_edit.html",
        {"form": form, "income": income},
    )


@login_required
def income_delete(request, pk):
    income = get_object_or_404(Income, pk=pk, user=request.user)

    if request.method == "POST":
        income.delete()
        messages.success(request, "Income entry deleted.")
        return redirect("bookkeeping:income_list")

    return render(
        request, "bookkeeping/income/income_confirm_delete.html", {"income": income}
    )


# ===========================
# EXPENSE VIEWS
# ===========================


@login_required
def expense_list(request):
    """List all expense entries for the logged-in user."""

    qs = Expense.objects.filter(user=request.user).select_related("category")

    # Search
    search = request.GET.get("search")
    if search:
        qs = qs.filter(
            Q(description__icontains=search) | Q(supplier_name__icontains=search)
        )

    # Quarter
    quarter = request.GET.get("quarter")
    if quarter:
        qs = qs.filter(quarter=quarter)

    # Category
    category_id = request.GET.get("category")
    if category_id:
        qs = qs.filter(category_id=category_id)

    # Date range
    date_from = request.GET.get("date_from")
    if date_from:
        qs = qs.filter(date__gte=date_from)

    date_to = request.GET.get("date_to")
    if date_to:
        qs = qs.filter(date__lte=date_to)

    # Order
    order_by = request.GET.get("order_by", "-date")
    qs = qs.order_by(order_by)

    # Totals
    totals = qs.aggregate(total=Sum("amount"), total_vat=Sum("vat_amount"))
    total_expenses = totals["total"] or 0
    total_vat = totals["total_vat"] or 0

    paginator = Paginator(qs, 20)
    page = paginator.get_page(request.GET.get("page"))

    context = {
        "expense_list": page,
        "total_expenses": total_expenses,
        "total_vat": total_vat,
        "expense_categories": Category.objects.filter(category_type="expense"),
        "quarters": (
            Expense.objects.filter(user=request.user)
            .values_list("quarter", flat=True)
            .distinct()
            .order_by("-quarter")
        ),
        # Returned filters
        "search_query": search,
        "filter_date_from": date_from,
        "filter_date_to": date_to,
        "filter_category": category_id,
        "filter_quarter": quarter,
        "order_by": order_by,
    }

    return render(request, "bookkeeping/expense/expense_list.html", context)


@login_required
def expense_create(request):
    if request.method == "POST":
        form = ExpenseForm(request.POST, request.FILES, user=request.user)
        if form.is_valid():
            obj = form.save(commit=False)
            obj.user = request.user
            obj.save()

            if "save_and_add" in request.POST:
                return redirect("bookkeeping:expense_create")
            return redirect("bookkeeping:expense_list")
    else:
        form = ExpenseForm(user=request.user)

    return render(request, "bookkeeping/expense/expense_form.html", {"form": form})


@login_required
def expense_detail(request, pk):
    expense = get_object_or_404(Expense, pk=pk, user=request.user)
    total = expense.amount + expense.vat_amount
    return render(
        request,
        "bookkeeping/expense/expense_detail.html",
        {"expense": expense, "total": total},
    )


@login_required
def expense_edit(request, pk):
    expense = get_object_or_404(Expense, pk=pk, user=request.user)

    if request.method == "POST":
        form = ExpenseForm(
            request.POST, request.FILES, instance=expense, user=request.user
        )
        if form.is_valid():
            obj = form.save(commit=False)
            obj.user = request.user
            obj.save()
            return redirect("bookkeeping:expense_detail", pk=expense.pk)
    else:
        form = ExpenseForm(instance=expense, user=request.user)

    return render(
        request,
        "bookkeeping/expense/expense_edit.html",
        {"form": form, "expense": expense},
    )


@login_required
def expense_delete(request, pk):
    expense = get_object_or_404(Expense, pk=pk, user=request.user)

    if request.method == "POST":
        expense.delete()
        messages.success(request, "Expense entry deleted.")
        return redirect("bookkeeping:expense_list")

    return render(
        request,
        "bookkeeping/expense/expense_confirm_delete.html",
        {"expense": expense},
    )


@login_required
def export_income_csv(request):
    response = HttpResponse(content_type="text/csv")
    response["Content-Disposition"] = "attachment; filename=income.csv"

    writer = csv.writer(response)
    writer.writerow(
        ["Date", "Description", "Client", "Net Amount", "VAT Amount", "Category"]
    )

    incomes = Income.objects.filter(user=request.user).order_by("-date")

    for item in incomes:
        writer.writerow(
            [
                item.date,
                item.description,
                item.client_name or "",
                item.amount,
                item.category.name if item.category else "",
            ]
        )

    return response


@login_required
def export_expense_csv(request):
    response = HttpResponse(content_type="text/csv")
    response["Content-Disposition"] = "attachment; filename=expenses.csv"

    writer = csv.writer(response)
    writer.writerow(
        ["Date", "Description", "Supplier", "Net Amount", "VAT Amount", "Category"]
    )

    expenses = Expense.objects.filter(user=request.user).order_by("-date")

    for item in expenses:
        writer.writerow(
            [
                item.date,
                item.description,
                item.supplier_name or "",
                item.amount,
                item.vat_amount,
                item.category.name if item.category else "",
            ]
        )

    return response
