from django.db import models
from django.contrib.auth.models import AbstractUser
from django.contrib.auth.base_user import BaseUserManager


# ==========================================================
# USER MODEL (Starter Edition)
# ==========================================================


class UserManager(BaseUserManager):
    use_in_migrations = True

    def _create_user(self, email, password, **extra_fields):
        """
        Create and save a User with the given email and password.
        """
        if not email:
            raise ValueError("Users must have an email address")

        email = self.normalize_email(email)
        user = self.model(email=email, **extra_fields)
        user.set_password(password)
        user.save(using=self._db)
        return user

    def create_user(self, email, password=None, **extra_fields):
        extra_fields.setdefault("is_staff", False)
        extra_fields.setdefault("is_superuser", False)
        return self._create_user(email, password, **extra_fields)

    def create_superuser(self, email, password=None, **extra_fields):
        extra_fields.setdefault("is_staff", True)
        extra_fields.setdefault("is_superuser", True)

        if extra_fields.get("is_staff") is not True:
            raise ValueError("Superuser must have is_staff=True.")
        if extra_fields.get("is_superuser") is not True:
            raise ValueError("Superuser must have is_superuser=True.")

        return self._create_user(email, password, **extra_fields)


class User(AbstractUser):
    """
    Clean, simple Local-Edition user model.
    Email = login.
    No subscriptions, no verification, no trial.
    """

    username = None
    email = models.EmailField(unique=True)

    first_name = models.CharField(max_length=150, blank=True)
    last_name = models.CharField(max_length=150, blank=True)

    # Optional bookkeeping personalisation
    phone_number = models.CharField(max_length=20, blank=True)
    utr_number = models.CharField(max_length=10, blank=True)
    nino = models.CharField(max_length=9, blank=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    USERNAME_FIELD = "email"
    REQUIRED_FIELDS = []

    objects = UserManager()

    def __str__(self):
        return self.email


# ==========================================================
# BUSINESS MODEL (Starter Edition)
# ==========================================================


class Business(models.Model):
    """
    Represents a business the user tracks locally.
    No HMRC integration. No external IDs.
    """

    BUSINESS_TYPE_CHOICES = [
        ("self-employment", "Self Employment"),
        ("uk-property", "UK Property"),
    ]

    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name="businesses")
    name = models.CharField(max_length=200)
    business_type = models.CharField(max_length=20, choices=BUSINESS_TYPE_CHOICES)
    trade_name = models.CharField(max_length=200, blank=True)

    accounting_period_start = models.DateField()
    accounting_period_end = models.DateField()

    cash_or_accruals = models.CharField(
        max_length=10,
        choices=[("cash", "Cash"), ("accruals", "Accruals")],
        default="cash",
    )

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ["name"]
        verbose_name_plural = "Businesses"

    def __str__(self):
        return f"{self.name} ({self.get_business_type_display()})"
